﻿using System.Collections.Generic;
using Hims.Shared.UserModels.Medications;
using Newtonsoft.Json;

namespace Hims.Api.Controllers
{
    using System;
    using System.Linq;
    using System.Threading.Tasks;
    using Domain.Helpers;
    using Domain.Services;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.UserModels;
    using Shared.UserModels.Filters;
    using Utilities;

    using Hims.Api.Helper;
    using Hims.Shared.Library.Enums;
    using Hims.Api.Models;

    /// <summary>
    /// The encounters controller.
    /// </summary>
    [Authorize]
    [Route("api/encounters")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class EncountersController : BaseController
    {
        /// <summary>
        /// the provider service
        /// </summary>
        private readonly IEncounterService encounterServices;

        /// <summary>
        /// The push notification helper.
        /// </summary>
        private readonly IPushNotificationHelper pushNotificationHelper;

        /// <summary>
        /// The account session services.
        /// </summary>
        private readonly IAccountSessionService accountSessionServices;

        /// <summary>
        /// the aes helper
        /// </summary>
        private readonly IAESHelper aesHelper;

        /// <summary>
        /// The appointments services.
        /// </summary>
        private readonly IAppointmentService appointmentsServices;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public EncountersController(
            IEncounterService encounterServices,
            IAppointmentService appointmentsServices,
            IAESHelper aesHelper,
            IAccountSessionService accountSessionServices,
            IPushNotificationHelper pushNotificationHelper,
            IAuditLogService auditLogServices)
        {
            this.encounterServices = encounterServices;
            this.appointmentsServices = appointmentsServices;
            this.pushNotificationHelper = pushNotificationHelper;
            this.aesHelper = aesHelper;
            this.accountSessionServices = accountSessionServices;
            this.auditLogServices = auditLogServices;
        }

        /// <summary>
        /// To find encounter
        /// </summary>
        /// <param name="model" >
        /// The encounter filter model.
        /// </param>
        /// <returns>
        /// The encounter model.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Encounter model.
        /// - 400 - Sorry! We don't have a encounter in the system.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [AllowAnonymous]
        [HttpPost]
        [Route("find")]
        [ProducesResponseType(typeof(EncounterModel), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindAsync([FromBody] EncounterFilterModel model)
        {
            model = (EncounterFilterModel)EmptyFilter.Handler(model);

            if (!string.IsNullOrEmpty(model.EncryptedEncounterId))
            {
                model.EncounterId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedEncounterId));
            }

            var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
            var response = await this.encounterServices.FindByAppointmentAsync(appointmentId);
            var providerId = !string.IsNullOrEmpty(model.EncryptedProviderId) ? Convert.ToInt32(this.aesHelper.Decode(model.EncryptedProviderId)) : 0;
            var patientId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedPatientId));
            var appointments = await this.appointmentsServices.FindByPatientAsync(patientId, providerId);
            var appointmentList = appointments.ToList();
            foreach (var item in appointmentList)
            {
                item.EncryptedAppointmentId = this.aesHelper.Encode(item.AppointmentId.ToString());
                item.EncryptedPatientId = model.EncryptedPatientId;
                item.AppointmentTimeString = Convert.ToDateTime(DateTime.Now.ToString("yyyy-MM-dd")).Add(item.AppointmentTime).ToString("hh:mm tt");
            }

            if (response != null && response.EncounterId > 0)
            {
                response.EncryptedEncounterId = this.aesHelper.Encode(response.EncounterId.ToString());
            }

            var oldAppointment = new PreviousAppointmentModel();
            if (response == null || response.EncounterId == 0)
            {
                oldAppointment = await this.appointmentsServices.FindPreviousAppointmentAsync(appointmentId);
                if (oldAppointment != null)
                {
                    oldAppointment.EncryptedAppointmentId = this.aesHelper.Encode(oldAppointment.AppointmentId.ToString());
                }
            }

            return this.Success(new { Dashboard = response, Appointments = appointmentList, OldAppointment = oldAppointment });
        }

        /// <summary>
        /// To find encounter
        /// </summary>
        /// <param name="model" >
        /// The encounter filter model.
        /// </param>
        /// <returns>
        /// The encounter model.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Encounter model.
        /// - 400 - Sorry! We don't have a encounter in the system.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [AllowAnonymous]
        [HttpPost]
        [Route("find-previous-encounter")]
        [ProducesResponseType(typeof(EncounterModel), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindEncounterAsync([FromBody] EncounterFilterModel model)
        {
            model = (EncounterFilterModel)EmptyFilter.Handler(model);

            if (!string.IsNullOrEmpty(model.EncryptedEncounterId))
            {
                model.EncounterId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedEncounterId));
            }

            var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
            var encounter = await this.encounterServices.FindByAppointmentAsync(appointmentId);

            return this.Success(encounter);
        }

        /// <summary>
        /// To find encounter
        /// </summary>
        /// <param name="model" >
        /// The encounter filter model.
        /// </param>
        /// <returns>
        /// The encounter model.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Encounter model.
        /// - 400 - Sorry! We don't have a encounter in the system.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [AllowAnonymous]
        [HttpPost]
        [Route("find-encounter")]
        [ProducesResponseType(typeof(EncounterModel), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindDashboardAsync([FromBody] EncounterFilterModel model)
        {
            model = (EncounterFilterModel)EmptyFilter.Handler(model);

            if (!string.IsNullOrEmpty(model.EncryptedEncounterId))
            {
                model.EncounterId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedEncounterId));
            }

            var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
            var encounter = await this.encounterServices.FindEncounterAsync(appointmentId, model.Type);

            return this.Success(encounter);
        }

        /// <summary>
        /// The add encounter.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Encounter added successfully.
        /// - 409 - Encounter already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [Authorize]
        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody] EncounterModel model,[FromHeader] LocationHeader header)
        {
            model = (EncounterModel)EmptyFilter.Handler(model);
            var response = await this.encounterServices.AddAsync(model);

            switch (response)
            {
                case -1:
                    return this.Conflict("Encounter has been exist with the selected appointment.");
                case 0:
                    return this.ServerError();
                default:
                    var basicDetails = await this.encounterServices.GetBasicAppointmentDetails(model.AppointmentId, model.IsAdmission);
                    if (string.IsNullOrEmpty(model.Medications))
                    {
                        await NotificationHelper.Notification(
                            basicDetails.PatientId,
                            Roles.Patient,
                            NotificationIntimate.FullTranscriptionAdded,
                            this.aesHelper.Encrypt(model.AppointmentId.ToString()),
                            this.accountSessionServices,
                            this.pushNotificationHelper);
                    }

                    if (!string.IsNullOrEmpty(model.Medications))
                    {
                        await NotificationHelper.Notification(
                            basicDetails.PatientId,
                            Roles.Patient,
                            NotificationIntimate.PrescriptionAdded,
                            this.aesHelper.Encrypt(model.AppointmentId.ToString()),
                            this.accountSessionServices,
                            this.pushNotificationHelper);
                    }

                    var auditLogModel = new AuditLogModel
                    {
                        AccountId = model.CreatedBy,
                        LogTypeId = (int)LogTypes.Encounters,
                        LogFrom = (int)AccountType.Provider,
                        LogDate = DateTime.UtcNow,
                        LogDescription = $"Encounter has been added.",
                        LocationId= Convert.ToInt32(header.LocationId)
                    };
                    await this.appointmentsServices.UpdateEncounterTypeAsync(model.AppointmentId, (int)EncounterTypes.Encounter, model.IsAdmission);
                    await this.auditLogServices.LogAsync(auditLogModel);
                    return this.Success(response.ToString());
            }
        }

        /// <summary>
        /// The update encounter.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Encounter updated successfully.
        /// - 409 - Encounter already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [Authorize]
        [HttpPost]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] EncounterModel model, [FromHeader] LocationHeader header)
        {
            model = (EncounterModel)EmptyFilter.Handler(model);
            var response = await this.encounterServices.UpdateAltAsync(model);
            switch (response.Response)
            {
                case -1:
                    return this.Conflict("Encounter has been exist with the selected appointment.");
                case 0:
                    return this.ServerError();
                default:
                    var basicDetails = await this.encounterServices.GetBasicAppointmentDetails(model.AppointmentId, model.IsAdmission);
                    if (response.Status == 1)
                    {
                        await NotificationHelper.Notification(
                            basicDetails.PatientId,
                            Roles.Patient,
                            NotificationIntimate.FullTranscriptionUpdated,
                            this.aesHelper.Encrypt(model.AppointmentId.ToString()),
                            this.accountSessionServices,
                            this.pushNotificationHelper);
                    }

                    if (response.Status == 2)
                    {
                        await NotificationHelper.Notification(
                            basicDetails.PatientId,
                            Roles.Patient,
                            NotificationIntimate.PrescriptionUpdated,
                            this.aesHelper.Encrypt(model.AppointmentId.ToString()),
                            this.accountSessionServices,
                            this.pushNotificationHelper);
                    }

                    var auditLogModel = new AuditLogModel
                    {
                        AccountId = model.ModifiedBy,
                        LogTypeId = (int)LogTypes.Encounters,
                        LogFrom = (int)AccountType.Provider,
                        LogDate = DateTime.UtcNow,
                        LogDescription = $"Encounter has been updated.",
                        LocationId= Convert.ToInt32(header.LocationId)
                    };
                    await this.auditLogServices.LogAsync(auditLogModel);
                    return this.Success("Encounter has been updated successfully.");
            }
        }

        /// <summary>
        /// The delete encounter.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Encounter deleted successfully.
        /// - 409 - Encounter can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [Authorize]
        [HttpDelete]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody] EncounterModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                model = (EncounterModel)EmptyFilter.Handler(model);
                var response = await this.encounterServices.DeleteAsync(model.EncounterId);
                if (response == 0)
                {
                    return this.ServerError();
                }
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.Encounters,
                    LogFrom = (int)AccountType.Provider,
                    LogDate = DateTime.UtcNow,
                    LogDescription = $"Encounter has been deleted.",
                    LocationId= Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);
                return this.Success("Encounter has been deleted successfully.");
            }
            catch (NpgsqlException exception)
            {
                if (exception.Message.Contains("violates foreign key constraint"))
                {
                    return this.Conflict("Encounter can't be deleted. Please contact administrator.");
                }

                return this.ServerError();
            }
        }

        /// <summary>
        /// The find patient async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [AllowAnonymous]
        [HttpPost]
        [Route("findPatient")]
        [ProducesResponseType(typeof(EncounterModel), 200)]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindPatientAsync([FromBody] EncounterFilterModel model)
        {
            model = (EncounterFilterModel)EmptyFilter.Handler(model);

            if (!string.IsNullOrEmpty(model.EncryptedEncounterId))
            {
                model.EncounterId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedEncounterId));
            }

            var response = await this.encounterServices.FindPatientAsync(model.EncounterId);

            return response == null ? this.BadRequest("Sorry! We don't have a encounter in the system") : this.Success(response);
        }

        /// <summary>
        /// The modify async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [AllowAnonymous]
        [HttpPost]
        [Route("modify")]
        [ProducesResponseType(typeof(EncounterModel), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModifyAsync([FromBody] EncounterModifyModel model,[FromHeader] LocationHeader header)
        {
            model = (EncounterModifyModel)EmptyFilter.Handler(model);
            var encounterId = model.EncounterId == 0
                                  ? await this.encounterServices.AddEncounterAltAsync(model)
                                  : await this.encounterServices.UpdateEncounterAltAsync(model);
            switch (encounterId.Response)
            {
                case -1:
                    return this.Conflict("Encounter has been exist with the selected appointment.");
                case 0:
                    return this.ServerError();
                default:
                    var basicDetails = await this.encounterServices.GetBasicAppointmentDetails(model.AppointmentId, model.IsAdmission);
                    if (encounterId.Status == 1 && model.EncounterId == 0)
                    {
                        await NotificationHelper.Notification(
                        basicDetails.PatientId,
                        Roles.Patient,
                        model.EncounterId == 0 ? NotificationIntimate.FullTranscriptionAdded : NotificationIntimate.FullTranscriptionUpdated,
                        this.aesHelper.Encrypt(model.AppointmentId.ToString()),
                        this.accountSessionServices,
                        this.pushNotificationHelper);
                    }

                    if (encounterId.Status == 2)
                    {
                        await NotificationHelper.Notification(
                        basicDetails.PatientId,
                        Roles.Patient,
                        model.EncounterId == 0 ? NotificationIntimate.PrescriptionAdded : NotificationIntimate.PrescriptionUpdated,
                        this.aesHelper.Encrypt(model.AppointmentId.ToString()),
                        this.accountSessionServices,
                        this.pushNotificationHelper);
                    }

                    var auditLogModel = new AuditLogModel
                    {
                        AccountId = model.ModifiedBy,
                        LogTypeId = (int)LogTypes.Encounters,
                        LogFrom = (int)AccountType.Provider,
                        LogDate = DateTime.UtcNow,
                        LogDescription = $"Encounter has been updated.",
                        LocationId= Convert.ToInt32(header.LocationId)
                    };
                    await this.appointmentsServices.UpdateEncounterTypeAsync(model.AppointmentId, (int)EncounterTypes.Encounter, model.IsAdmission);
                    await this.auditLogServices.LogAsync(auditLogModel);
                    return this.Success(encounterId.Response);
            }
        }

        /// <summary>
        /// The find full transcript async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [AllowAnonymous]
        [HttpPost]
        [Route("find-full-transcript")]
        [ProducesResponseType(typeof(InternalMedicineFullTranscriptModel), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FindFullTranscriptAsync([FromBody] EncounterFilterModel model)
        {
            model = (EncounterFilterModel)EmptyFilter.Handler(model);
            var appointmentId = Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId));
            var encounterFullTranscript = await this.encounterServices.FindFullTranscriptAsync(appointmentId);
            if (encounterFullTranscript == null || encounterFullTranscript.EncounterId == 0)
            {
                return this.BadRequest("Sorry! We don't have a encounter in the system.");
            }

            encounterFullTranscript.AppointmentTimeString = Convert.ToDateTime(DateTime.Now.ToString("yyyy-MM-dd")).Add(encounterFullTranscript.AppointmentTime).ToString("hh:mm tt");
            return this.Success(encounterFullTranscript);
        }

        /// <summary>
        /// To find encounter
        /// </summary>
        /// <param name="model" >
        /// The encounter filter model.
        /// </param>
        /// <returns>
        /// The encounter model.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Encounter model.
        /// - 400 - Sorry! We don't have a encounter in the system.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [AllowAnonymous]
        [HttpPost]
        [Route("fetch-medications")]
        [ProducesResponseType(typeof(EncounterModel), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchMedicationsAsync([FromBody] MedicationsRequestModel model)
        {
            model = (MedicationsRequestModel)EmptyFilter.Handler(model);
            var record = new MedicationsViewModel
            {
                Medications = new List<MedicinesModel>()
            };

            var data = await this.encounterServices.FetchActiveMedications(model.PatientId);
            if (data == null)
            {
                return this.Success(record);
            }

            record.AppointmentId = data.AppointmentId;
            record.CreatedDate = data.CreatedDate;
            record.ModifiedDate = data.ModifiedDate;
            record.Medications = JsonConvert.DeserializeObject<List<MedicinesModel>>(data.Medications);

            return this.Success(record);
        }

        /// <summary>
        /// To find appointment
        /// </summary>
        /// <param name="model" >
        /// The appointment find request Model.
        /// </param>
        /// <returns>
        /// The appointment model.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - appointment model.
        /// - 400 - Sorry! We don't have a appointment in the system.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [AllowAnonymous]
        [Route("fetch-vitals")]
        [ProducesResponseType(typeof(IEnumerable<VitalsModel>), 200)]
        [ProducesResponseType(typeof(string), 400)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchVitals([FromBody] EncounterFilterModel model)
        {
            model = (EncounterFilterModel)EmptyFilter.Handler(model);
            model.AppointmentId = model.AppointmentId <= 0 ? Convert.ToInt32(this.aesHelper.Decode(model.EncryptedAppointmentId)) : model.AppointmentId;

            var vitals = await this.encounterServices.FetchVitals(model);

            if (vitals == null)
            {
                return this.BadRequest("Sorry! We don't have a appointment in the system.");
            }

            foreach (var item in vitals)
            {
                item.Vital = JsonConvert.DeserializeObject<Vital>(item.Vitals);
            }

            return this.Success(vitals);
        }
    }
}